/*=======================================================================================

 M24XXX: Serial I2C Bus EEPROM Driver for I2C Bus EEPROM Verilog Simulation Model

=========================================================================================

 This program is provided "as is" without warranty of any kind, either
 expressed or implied, including but not limited to, the implied warranty
 of merchantability and fitness for a particular purpose. The entire risk
 as to the quality and performance of the program is with you. Should the
 program prove defective, you assume the cost of all necessary servicing,
 repair or correction.
 
 Copyright 2012, STMicroelectronics Corporation, All Right Reserved.

=======================================================================================*/


/*=======================================================================================
                                   REVISION HISTORY

rev1.0 (29 Jan 2013)
- first release
rev1.1 (19 Nov 2013)
- current and random read: fix for the automatic address increment at the end
of the transaction.
=======================================================================================*/


/*=====================================
 Serial I2C Bus EEPROM Driver
=====================================*/

`include "M24XXX_Macro.v"
`include "M24XXX_Parameters.v"

//-------------------------------------
module M24XXX_DRV(
                    Ei,
                    SCL,
                    SDA,
                    WC,
                    VCC
                 );

//-------------------------------------
inout  SDA;
output SCL;
output WC;
output VCC;
output[`Ei_Inputs-1:0] Ei;

//-------------------------------------
reg WC,VCC;
reg[`Ei_Inputs-1:0] Ei;
reg [2:0] addr1s;
//-------------------------------------

Stimuli stimuli(.SCL(SCL),.SDA(SDA));

//-------------------------------------
initial begin
       if (`M2Kb_var || `M1Kb_var)
          addr1s= 3'b000; 
       else if (`M4Kb_var ||`M1Mb_var )
          addr1s= 3'b001;
       else if (`M8Kb_var || `M2Mb_var)
          addr1s= 3'b011;
       else if (`M16Kb_var)
          addr1s= 3'b111;
       else
          addr1s= 3'b000;

        VCC = 1'b0;
        #200;

        $display("============================================");
        $display("%t: TESTING1: Device is not powered on.",$realtime);
        $display("============================================");
        Ei = 3'b000;
        stimuli.Write_in_data(1,{4'b1010,addr1s},{`Address_Bits {1'b1}},8'hA5);//`Address_Bits'hffff
                      //write 1 byte, dev_sel_code is 1010000, address=ffffh, write data is A5h
        #`tWR;        //write 1 byte in Page511

        stimuli.Random_address_read(1,{4'b1010,addr1s},{`Address_Bits {1'b1}});
                      //read 1 byte, dev_sel_code is 1010000, address=ffffh
                      //read out data in address=ffffh

        stimuli.Current_address_read(1,7'b1010000);
                      //read 1 byte, dev_sel_code is 1010000
                      //read out data in address 0000h

        VCC = 1'b1;
//=========================================================
//check memory operation with various chip enable set
//=========================================================
        $display("=================================================================");
        $display("%t: TESTING2: Memory operation with various chip enable set.",$realtime);
        $display("=================================================================");


//`ifdef MEM_32K_to_2M
case (`Ei_Inputs)
3: begin
        Ei = 3'b000;
        stimuli.Random_address_read(1,7'b1010000,{`Address_Bits {1'b0}});  //`Address_Bits'h0000
        Ei = 3'b001;
        stimuli.Random_address_read(1,7'b1010001,{{`Address_Bits-1 {1'b0}},1'b1}); //`Address_Bits'h0001
        Ei = 3'b010;
        stimuli.Random_address_read(1,7'b1010010,{{`Address_Bits-2 {1'b0}},2'b10}); //`Address_Bits'h0002
        Ei = 3'b011;
        stimuli.Random_address_read(1,7'b1010011,{{`Address_Bits-2 {1'b0}},2'b11}); //`Address_Bits'h0003
        Ei = 3'b100;
        stimuli.Random_address_read(1,7'b1010100,{{`Address_Bits-4 {1'b1}},4'b1100}); //`Address_Bits'hfffc
        Ei = 3'b101;
        stimuli.Random_address_read(1,7'b1010101,{{`Address_Bits-4 {1'b1}},4'b1101}); //`Address_Bits'hfffd
        Ei = 3'b110;
        stimuli.Random_address_read(1,7'b1010110,{{`Address_Bits-4 {1'b1}},4'b1110}); //`Address_Bits'hfffe
        Ei = 3'b111;
        stimuli.Random_address_read(1,7'b1010111,{`Address_Bits {1'b1}}); //`Address_Bits'hffff
        Ei = 3'b111;
        stimuli.Random_address_read(1,7'b1010110,{`Address_Bits {1'b0}});  //`Address_Bits'h0000
end
2: begin //M4Kb M1Mb
//`elsif MEM_1K_to_16K

        Ei = 2'b00;
        stimuli.Random_address_read(1,7'b1010000,`Address_Bits'h01);
        stimuli.Random_address_read(1,7'b1010001,`Address_Bits'hfc);
        Ei = 2'b01;
        stimuli.Random_address_read(1,7'b1010010,`Address_Bits'h01);
        stimuli.Random_address_read(1,7'b1010011,`Address_Bits'hfd);
        Ei = 2'b10;
        stimuli.Random_address_read(1,7'b1010100,`Address_Bits'h02);
        stimuli.Random_address_read(1,7'b1010101,`Address_Bits'hfe);
        Ei = 2'b11;
        stimuli.Random_address_read(1,7'b1010110,`Address_Bits'h03);
        stimuli.Random_address_read(1,7'b1010111,`Address_Bits'hff);
        Ei = 2'b11;
        stimuli.Random_address_read(1,7'b1010100,`Address_Bits'h00);
        stimuli.Random_address_read(1,7'b1010011,`Address_Bits'hff);
end
1: begin //M8Kb M2Mb
        Ei = 1'b0;
        stimuli.Random_address_read(1,7'b1010000,`Address_Bits'h00);
        stimuli.Random_address_read(1,7'b1010011,`Address_Bits'hfe);
        Ei = 1'b1;
        stimuli.Random_address_read(1,7'b1010100,`Address_Bits'h01);
        stimuli.Random_address_read(1,7'b1010111,`Address_Bits'hff);
        Ei = 1'b1;
        stimuli.Random_address_read(1,7'b1010000,`Address_Bits'h00);
        stimuli.Random_address_read(1,7'b1010011,`Address_Bits'hff);
        Ei = 1'b0;
        stimuli.Random_address_read(1,7'b1010100,`Address_Bits'h00);
        stimuli.Random_address_read(1,7'b1010111,`Address_Bits'hff);
end
0: begin //M16Kb
        stimuli.Random_address_read(1,7'b1010000,`Address_Bits'h00);
        stimuli.Random_address_read(1,7'b1010111,`Address_Bits'hfe);

        stimuli.Random_address_read(1,7'b1010000,`Address_Bits'h01);
        stimuli.Random_address_read(1,7'b1010111,`Address_Bits'hff);

        stimuli.Random_address_read(1,7'b1100000,`Address_Bits'h00);
        stimuli.Random_address_read(1,7'b1000111,`Address_Bits'hff);
end
endcase
//`endif

`ifndef MEM_1K_to_16K
if (`IDPAGE) begin
        $display("=================================================================");
        $display("%t: TESTING3: ID MEMORY PAGE.",$realtime);
        $display("=================================================================");
        WC = 1'b0;    // write enable
        Ei = 3'b000;

        stimuli.Read_lock_status(7'b1011000,8'hC8);

        //stimuli.Start_Stop;


        stimuli.Write_in_data(`Page_Size-1,7'b1011000,{{`Address_Bits-4 {1'b0}},4'h2},8'h5A);
                      //dev_sel_code is 1010000, address=ffffh, write data is 5Ah
        #`tWR;        //address counter=ffffh after write completely


        stimuli.Current_address_read(2,7'b1011000);
                      //read 2 bytes,dev_sel_code is 1010000
                      //read out data in address ffffh and 0000h

        stimuli.Random_address_read(2,{7'b1011000},`Page_Size-1);
                      //read 2 bytes,dev_sel_code is 1010000,address=Page_Size-1
        stimuli.Random_address_read(3,{7'b1011000},{`Address_Bits {1'b0}});
                      //read 2 bytes,dev_sel_code is 1010000,address=0


        //send the lock id page
       // stimuli.Write_in_data(1,7'b1011000,`Address_Bits'd1024,8'hA2);
        stimuli.Write_lock_status(3'b101);
        #`tWR;        
        //this page lock is ignored cause it's already locked
        stimuli.Write_lock_status(3'b101);
        #`tWR;        
        stimuli.Current_address_read(2,7'b1011000);

         //write ignored cause page is locked
        stimuli.Write_in_data(`Page_Size,7'b1011000,{`Address_Bits {1'b0}},8'hB6);
                      // dev_sel_code is 1010000, address=ffffh, write data is B6h
        #`tWR;        

        //check if B6h was written, it should not be cause the ID page is locked
        stimuli.Current_address_read(3,7'b1011000);

        stimuli.Random_address_read(3,7'b1011000,{`Address_Bits {1'b0}});
                      //read 3 bytes,dev_sel_code is 1010000,address=ff7fh

        //check id lock status
        stimuli.Read_lock_status(7'b1011000,8'hC8);
end
else `endif
begin
        $display("=================================================================");
        $display("%t: TESTING3: NO ID MEMORY PAGE.",$realtime);
        $display("=================================================================");
end


//===============================================
//check roll-over for read operation
//===============================================

        $display("=================================================================");
        $display("%t: TESTING4: memory read operation and address roll over.",$realtime);
        $display("=================================================================");
        WC = 1'b0;    // write enable
        Ei = 3'b000;
        stimuli.Write_in_data(`Page_Size,{4'b1010,addr1s},{`Address_Bits {1'b1}},8'hA5); //`Address_Bits'hffff
                      //write 128 bytes, dev_sel_code is 1010000, address=ffffh, write data is A5h
        #`tWR;        //write 128 bytes in Page511, address counter=ffffh after write completely

        stimuli.Current_address_read(2,7'b1010000);
                      //read 2 bytes,dev_sel_code is 1010000
                      //read out data in address ffffh and 0000h

        stimuli.Random_address_read(2,{4'b1010,addr1s},{`Address_Bits {1'b1}}); //`Address_Bits'hffff
                      //read 2 bytes,dev_sel_code is 1010000,address=ffffh
                      //read out data in address=ffffh and 0000h

        stimuli.Random_address_read(2,{4'b1010,addr1s},((`Memory_Size-1)-`Page_Size)); //`Address_Bits'hff7f  
                      //read 2 bytes,dev_sel_code is 1010000,address=ff7fh
                      //read out data in address=ff7fh and ff80h

 
//===============================================
//check roll-over in page for write operation
//===============================================
        $display("=================================================================");
        $display("%t: TESTING4: Memory write operation and address roll-over.",$realtime);
        $display("=================================================================");
        WC = 1'b0;      //write enable
        Ei = 3'b000;
        stimuli.Write_in_data(`Page_Size,7'b1010000,((`Page_Size*2)-1),8'hA5); //`Address_Bits'h00ff
                        //write 128 bytes, dev_sel_code is 1010000, address=00ffh(page1), write data is A5h
        #`tWR;

        stimuli.Random_address_read(`Page_Size+2,7'b1010000,(`Page_Size-1));  //`Address_Bits'h007f
                        //read 130 bytes, dev_sel_code is 1010000, address=007f~0100h


//==================================================
//check /WC=1,write inhibited
//==================================================
        $display("=================================================================");
        $display("%t: TESTING5: memory write operation with /WC enable or disable.",$realtime);
        $display("=================================================================");
        //------------------------/WC = 1
        Ei = 3'b000;
        WC = 1'b1;      //write disable
        stimuli.Write_in_data(`Page_Size,7'b1010000,{`Address_Bits {1'b0}},8'h66); //`Address_Bits'h0000
                        //write 128 bytes,dev_sel_code is 1010000,address=0000h,write data is 66h
        #`tWR;
        stimuli.Random_address_read(`Page_Size,7'b1010000,{`Address_Bits {1'b0}});

        //------------------------"1" glitch on /WC before start
        WC = 1'b1;
        #10;
        WC = 1'b0;      //write enable
        stimuli.Write_in_data(`Page_Size,7'b1010000,{`Address_Bits {1'b0}},8'h88);
                        //write 128 bytes,dev_sel_code is 1010000,address=0000h,write data is 88h
        #`tWR;
        stimuli.Random_address_read(`Page_Size,7'b1010000,{`Address_Bits {1'b0}});

        //------------------------"1" glitch on /WC bewteen start and end of the low byte of address
        stimuli.I2C_Start;
        WC = 1'b1;      //write disable
        #10;
        WC = 1'b0;
        stimuli.Dev_Sel_Code_W(7'b1010000);         //Send first device select code with bit0=0
        stimuli.Wait_Ack;
`ifndef MEM_1K_to_16K
        stimuli.Write_add_h({`Address_Bits {1'b0}});   //Send high byte of address
        stimuli.Wait_Ack;
`endif
        stimuli.Write_add_l({`Address_Bits {1'b0}});   //Send low byte of address
        stimuli.Wait_Ack;
        stimuli.Write_data(`Page_Size,8'h99);              //send data byte to program
        stimuli.I2C_Stop;
        #`tWR;
        stimuli.Random_address_read(`Page_Size,7'b1010000,{`Address_Bits {1'b0}});

        //-----------------------"1" glitch on /WC after low byte of address
        stimuli.I2C_Start;
        stimuli.Dev_Sel_Code_W(7'b1010000);         //Send first device select code with bit0=0
        stimuli.Wait_Ack;
`ifndef MEM_1K_to_16K
        stimuli.Write_add_h({`Address_Bits {1'b0}});   //Send high byte of address
        stimuli.Wait_Ack;
`endif
        stimuli.Write_add_l({`Address_Bits {1'b0}});   //Send low byte of address
        stimuli.Wait_Ack;
        WC = 1'b1;
        #10;
        WC = 1'b0;      //write enable
        stimuli.Write_data(`Page_Size,8'haa);               //send data byte to program
        stimuli.I2C_Stop;
        #`tWR;
        stimuli.Random_address_read(`Page_Size,7'b1010000,{`Address_Bits {1'b0}});


//=========================================================
        VCC = 1'b0;
        #200;
        VCC = 1'b1;
        #200;
        $display("======================================================================================");
        $display("%t: TESTING6: memory write operation while internal write cycle is in progress.",$realtime);
        $display("======================================================================================");
        Ei = 3'b000;
        stimuli.Write_in_data(1,7'b1010000,{`Address_Bits {1'b1}},8'hA5);
                      //write 1 byte, dev_sel_code is 1010000, address=ffffh, write data is A5h
        stimuli.I2C_Start;
        stimuli.I2C_Stop;
        stimuli.Random_address_read(1,{4'b1010,addr1s},{`Address_Bits {1'b1}});
                      //read 1 byte,dev_sel_code is 1010000
                      //read out data in address ffffh
        stimuli.Current_address_read(1,7'b1010000);
        
        stimuli.Write_in_data(1,{4'b1010,addr1s},{`Address_Bits {1'b1}},8'h5A);
                      //write 1 byte, dev_sel_code is 1010000, address=ffffh, write data is 5Ah
        #`tWR;
        stimuli.Random_address_read(1,{4'b1010,addr1s},{`Address_Bits {1'b1}});
                      //read 1 byte,dev_sel_code is 1010000
                      //read out data in address ffffh

//        $finish;
end

endmodule
